--[[----------------------------------------------------------------------------

LeicaTetherTask.lua
Tasks for Lightroom sample Tether plugin

--------------------------------------------------------------------------------

ADOBE SYSTEMS INCORPORATED
 Copyright 2007-2008 Adobe Systems Incorporated
 All Rights Reserved.

NOTICE: Adobe permits you to use, modify, and distribute this file in accordance
with the terms of the Adobe license agreement accompanying it. If you have received
this file from a source other than Adobe, then your use, modification, or distribution
of it requires the prior written permission of Adobe.

------------------------------------------------------------------------------]]

local LrLogger = import 'LrLogger'
local LrPathUtils = import 'LrPathUtils'
--local LrTasks = import 'LrTasks'
--local LrFunctionContext = import 'LrFunctionContext'
local LrFileUtils = import "LrFileUtils"


local LrRemoteCommunication = import 'LrRemoteCommunication'

local LrTableUtils = import "LrTableUtils"

require "LeicaTetherUtils"

--============================================================================--

LeicaTetherTask = {}

-------------------------------------------------------------------------------

local logger = LrLogger 'LeicaTetherTask'
	-- not enabled by default

--local ddt = import 'LrTableUtils'.debugDumpTable
local launchAndConnectToLeicaTetherServer

-------------------------------------------------------------------------------

function LeicaTetherTask.init( propertyTable, pluginPath, completionCallback )
	-- Attempt to find and launch the tether application
	
	logger:trace( "plugin path: ", pluginPath, " callback: ", completionCallback )
	
	LeicaTetherTask._connectedToPluginApp = false
	LeicaTetherTask._sentListDevicesMessage = false
	LeicaTetherTask._propertyTable = propertyTable
	
	launchAndConnectToLeicaTetherServer( pluginPath, completionCallback )
end

-------------------------------------------------------------------------------

function launchAndConnectToLeicaTetherServer( pluginPath, completionCallback )
	-- Attempt a connection
	
	local taskPath
	if MAC_ENV then
	
		if DEBUG then
			local testPath
			testPath = LrPathUtils.child( pluginPath, "../../../../../tether_leica/Debug/tether_leica.app/Contents/MacOS/tether_leica" )
			logger:trace( "debug build - testing: ", testPath )
			if LrFileUtils.exists( testPath ) == "file" then
				taskPath = testPath
			end
		end
		
		if not taskPath then
			taskPath = LrPathUtils.child( pluginPath, "Contents/PlugIns/tether_leica.app/Contents/MacOS/tether_leica" )
		end
	--	taskPath = pluginPath .. "/Contents/MacOS/tether_leica"
		
		logger:trace( "launching plugin app with path: ", taskPath )
	else
		taskPath = pluginPath .. "/tether_leica.exe"
		
		logger:trace( "launching plugin app with path: ", taskPath )
	end

	LrRemoteCommunication.spawnTaskAndConnect( "tether_leica", taskPath,
		function( serverRef )
			logger:trace( "launchAndConnectToLeicaTetherServer - serverRef: ", serverRef )
			LeicaTetherTask._serverConnection = serverRef
		
			-- Uncomment this to halt after starting up the task, giving you a chance to attach the
			-- debugger.
		--	 halt()
	
			if completionCallback then completionCallback( serverRef ~= nil ) end
		end )
	
	
end

-------------------------------------------------------------------------------

function LeicaTetherTask.shutdown( propertyTable )
	-- Send a message to the server asking it to shut down
	if LeicaTetherTask._serverConnection then
		logger:trace( "LeicaTetherTask.shutdown" )
		LrRemoteCommunication.closeNamedConnection( LeicaTetherTask._serverConnection, "tether_leica" )
	end
end

-------------------------------------------------------------------------------

function LeicaTetherTask.setCallback( propertyTable, callback )
	LeicaTetherTask._hostCallback = callback
end

-------------------------------------------------------------------------------

--local function getCaptureInfoCallback( captureId, messageType, result )
--	local device = LeicaTetherTask._connectedDevice
--	local capture = device.captures[ captureId ]
--	
--	logger:trace( 'getCaptureInfoCallback: Received capture info, now starting file download.' )
--	
--	if messageType == "ok" then
--		capture.captureInfo = result
--	elseif messageType == "error" then
--		capture.captureInfoErr = result
--	end
--	
--	LeicaTetherTask._hostCallback( "fileInfoAvailable", 
--						{ captureId = captureId, 
--							fileType = LrPathUtils.extension( result.szFileName ),
--							fileLength = result.size, } )
--
--	LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "startDownload", { capture_id = captureId }, 
--				function( messageType, result ) if messageType == "error" then capture.errorCode = result end end )
--end

-------------------------------------------------------------------------------

--local function requestThumbnailCallback( captureId, messageType, result )
--	local device = LeicaTetherTask._connectedDevice
--	local capture = device.captures[ captureId ]
--	
--	logger:tracef( 'requestThumbnailCallback: Received thumbnail message: %s, now requesting capture info.', messageType )
--	
--	if messageType == 'ok' then
--		LeicaTetherTask._hostCallback( "thumbnailAvailable", { captureId = captureId, thumbnailData = result.thumbnail_data } )
--	end
--	
--	LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "getCaptureInfo", { capture_id = captureId }, 
--				function( messageType, result ) getCaptureInfoCallback( captureId, messageType, result ) end )
--	capture.requestedInfo = true
--end

-------------------------------------------------------------------------------

function LeicaTetherTask.tetherCallback( propertyTable, messageType, result )
	logger:trace( "LeicaTetherTask.tetherCallback - message: ", messageType )

	if messageType == "objectWasAdded" then
		if LeicaTetherTask._hostCallback then
			local device = LeicaTetherTask._connectedDevice
			if not device then
				logger:error( "LeicaTetherTask.tetherCallback: Could not locate device." )
				return
			end
			local captureId = result.capture_id
			if not device.captures then
				device.captures = {}
			end
			if not device.captures[ captureId ] then
				device.captures[ captureId ] = {}
			end
			
			logger:tracef( "objectWasAdded - device %s, device.captures = %s, device.captures[%s] = %s", device, device.captures, captureId, device.captures[captureId] )
			
--			local capture = device.captures[ captureId ]
			
			-- Immediately request info about the capture and start the thumbnail download
--			LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "downloadThumbnail", { capture_id = captureId }, 
--						function( messageType, result ) requestThumbnailCallback( captureId, messageType, result ) end )
--			capture.requestedThumbnail = true

--			LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "getCaptureInfo", { capture_id = captureId }, 
--						function( messageType, result ) getCaptureInfoCallback( captureId, messageType, result ) end )
--			capture.requestedInfo = true
			
			logger:trace( "objectWasAdded - captureId = ", captureId )
			
			if result.has_sidecar and result.has_sidecar ~= 0 then
				result.primary_filename = LrPathUtils.replaceExtension( result.file_name, "DNG" )
			end
			
			LeicaTetherTask._hostCallback( "objectWasAdded", result )
		end
	elseif messageType == "dataIsAvailable" then
--		if LeicaTetherTask._hostCallback then
--			local device = LeicaTetherTask._connectedDevice
--			
--			if not device then
--				err( "LeicaTetherTask.tetherCallback: Could not locate device." )
--				return
--			end
--			local captureId = result.capture_id
--			local capture = device.captures[ captureId ]
--			
--			if not capture then
--				err( "LeicaTetherTask.tetherCallback: could not find capture id: ", tostring( captureId ) )
--				return
--			end
--			
--			LeicaTetherTask._hostCallback( "dataIsAvailable", 
--							{ captureId = captureId, 
--								imageData = string.sub( result.image_data, 1, result.data_size ), 
--								isDone = result.is_done, 
--								status = result.is_done and "done" or "ok" } )
--		end
	elseif messageType == "fileDownloadComplete" then
		if LeicaTetherTask._hostCallback then
			logger:trace( "LeicaTetherTask.tetherCallback - fileDownloadComplete: ", result.file_path )
			local device = LeicaTetherTask._connectedDevice
			
			if not device then
				logger:error( "LeicaTetherTask.tetherCallback: Could not locate device." )
				return
			end
			local captureId = result.capture_id
			local capture = device.captures[ captureId ]
			
			if not capture then
				logger:error( "LeicaTetherTask.tetherCallback: could not find capture id: ", captureId )
				return
			end
			
			LeicaTetherTask._hostCallback( "fileDownloadComplete", 
							{ captureId = captureId, 
								filePath = result.file_path, 
								fileType = LrPathUtils.extension( result.file_path ),
								status = "done" } )
		end
	elseif messageType == "deviceAdded" then
		logger:trace( "LeicaTetherTask.tetherCallback - deviceAdded: ", result.device_id )
		if LeicaTetherTask._propertyTable.availableDevices then
			local forceUpdate = false
			for i, device in ipairs( LeicaTetherTask._propertyTable.availableDevices ) do
				if device.device_id == result.device_id then
					forceUpdate = true
				end
			end
			if not forceUpdate then
				table.insert( LeicaTetherTask._propertyTable.availableDevices, result )
			end
		--	LrTableUtils.debugDumpTable( LeicaTetherTask._propertyTable.availableDevices )
			logger:trace( "- deviceAdded forceUpdate: ", forceUpdate )
			LeicaTetherTask._hostCallback( "availableDevicesChanged", forceUpdate )
			
		end
	elseif messageType == "deviceRemoved" then
		logger:trace( "LeicaTetherTask.tetherCallback - deviceRemoved: ", result.device_id )
		local deviceId = result.device_id
		if deviceId and LeicaTetherTask._propertyTable.availableDevices then
			for i, v in ipairs( LeicaTetherTask._propertyTable.availableDevices ) do
				if v.device_id == deviceId then
					table.remove( LeicaTetherTask._propertyTable.availableDevices, i )
					LeicaTetherTask._hostCallback( "availableDevicesChanged" )
					break
				end
			end
		end
	elseif messageType == "cameraSettingsChanged" then
		
	--	LrTableUtils.debugDumpTable( result, "Camera Settings - before processing" )
		local settings = LeicaTetherUtils.postProcessCameraSettings( result )
		LeicaTetherTask._hostCallback( "cameraSettingsChanged", settings )
	else
		LeicaTetherTask._hostCallback( messageType, result )
	end
end

-------------------------------------------------------------------------------


function LeicaTetherTask.pollForMessages( propertyTable )
--	logger:trace( 'LeicaTetherTask.pollForMessages()' )
	pcall( function()
		local message, params = LrRemoteCommunication.pollForMessage( LeicaTetherTask._serverConnection )
		
		if message then
			logger:trace( 'LeicaTetherTask.pollForMessages() - received message: ', message )
			
			LeicaTetherTask.tetherCallback( propertyTable, message, params )
		end
	end )
end

-------------------------------------------------------------------------------

function LeicaTetherTask.startDeviceSearch( propertyTable )
	
	logger:trace( "LeicaTetherTask.startDeviceSearch" )
	
	if LeicaTetherTask._connectedToPluginApp then return end
	
	if LeicaTetherTask._serverConnection then
		LeicaTetherTask._connectedToPluginApp = true
		LeicaTetherTask._sentListDevicesMessage = false

		logger:trace( 'LeicaTetherTask.startDeviceSearch - connection established.' )
		
--		LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "provideCallbackForNotifications", {}, 
--								function( messageType, result ) LeicaTetherTask.tetherCallback( propertyTable, messageType, result ) end )
								
		LeicaTetherTask.queryDevices( propertyTable )
	end
end

-------------------------------------------------------------------------------

function LeicaTetherTask.stopDeviceSearch( propertyTable )
	logger:trace( 'LeicaTetherTask.stopDeviceSearch' )
end

-------------------------------------------------------------------------------

function LeicaTetherTask.queryDevices( propertyTable )
	logger:trace( 'LeicaTetherTask.queryDevices' )
	
--	if not LeicaTetherTask._connectedToPluginApp then
--		LeicaTetherTask.startDeviceSearch( propertyTable )
--	end
	
	-- If we're still not connected, wait until later
	if not LeicaTetherTask._connectedToPluginApp then return end
	logger:trace( "LeicaTetherTask.queryDevices" )
	
	local function listDevicesCallback( messageType, result )
	
		logger:tracef( "listDevicesCallback called with result: %s", messageType )
	
		if messageType == "ok" then
			LeicaTetherTask._propertyTable.availableDevices = result and result.devices or {}
			LeicaTetherTask._hostCallback( "availableDevicesChanged" )
			
			LrTableUtils.debugDumpTable( LeicaTetherTask._propertyTable.availableDevices, "Device Table" )
		end
	end
	
	if not LeicaTetherTask._sentListDevicesMessage then
		logger:trace( 'LeicaTetherTask.queryDevices: sending message to server with callback:', listDevicesCallback )
		LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "listDevices", {}, listDevicesCallback )
		LeicaTetherTask._sentListDevicesMessage = true
	end
	
	if LeicaTetherTask._propertyTable.availableDevices then
--		local devices = {}
--		
--		for i, v in ipairs( LeicaTetherTask._propertyTable.availableDevices ) do
--			table.insert( devices, { id = v.device_id, name = v.name } )
--		end
--		
--		return { status = 'ok', devices = devices }
		return { status = 'ok' }
	else
		return { status = 'notReady' }
	end
end

-------------------------------------------------------------------------------

function LeicaTetherTask.connectDevice( propertyTable, deviceId, callback )
	logger:trace( 'LeicaTetherTask.connectDevice' )
	
	local function openDeviceCallback( messageType, result )
		if messageType == "ok" then
			LeicaTetherTask._sessionId = result and result._sessionId
			if callback and result and result.camera_name then
				LeicaTetherTask._connectedDevice.name = result.camera_name
				callback( result.camera_name )
			end
		end
	end
	
	for i,v in ipairs( LeicaTetherTask._propertyTable.availableDevices ) do
		if v.device_id == deviceId then
			LeicaTetherTask._connectedDevice = v
		end
	end

	LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "connectDevice", { device_id = deviceId }, openDeviceCallback )
	
	return { status = 'ok' }
end

-------------------------------------------------------------------------------

function LeicaTetherTask.disconnectDevice( propertyTable, deviceId )
	logger:trace( 'LeicaTetherTask.disconnectDevice' )
	
	LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "disconnectDevice", { device_id = deviceId }, nil )
end

-------------------------------------------------------------------------------

function LeicaTetherTask.doCapture( propertyTable, deviceId )
	logger:trace( 'LeicaTetherTask.doCapture' )
	
	local function doCaptureCallback( messageType, result )
		if messageType == "ok" then
		end
	end
	
	LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "doCapture", { device_id = deviceId }, doCaptureCallback )
	
	return { status = 'ok' }
end

-------------------------------------------------------------------------------

function LeicaTetherTask.downloadThumbnail( propertyTable, deviceId, captureId )
	logger:trace( 'LeicaTetherTask.downloadThumbnail' )

--	local device = LeicaTetherTask._connectedDevice
--	local capture = device and device.captures[ captureId ]
	
--	if not capture then
--		logger:errorf( "LeicaTetherTask.downloadThumbnail: capture table for captureId %s could not be found.", captureId )
--		return {
--			status = "error",
--			errorMsg = "Could not download thumbnail for capture id: " .. tostring(captureId),
--		}
--	end
--	
--	if capture.thumbnailData then
--		logger:trace( 'LeicaTetherTask.downloadThumbnail: sending thumbnail data back' )
--		
--		-- We've already gotten the data
--		return {
--			status = "ok",
--			thumbnailData = capture.thumbnailData.thumbnail_data,
--		}
--	end
--	
--	if not capture.requestedThumbnail then
--		LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "downloadThumbnail", { capture_id = captureId }, 
--						function( messageType, result ) requestThumbnailCallback( capture, messageType, result ) end )
--		capture.requestedThumbnail = true
--	end
	
	return {
		status = "notReady",
	}
end

-------------------------------------------------------------------------------

function LeicaTetherTask.getCaptureInfo( propertyTable, deviceId, captureId )
	logger:trace( 'LeicaTetherTask.getCaptureInfo' )

--	return { status = 'ok', fileType = 'jpg', fileLength = 100 }

	local device = LeicaTetherTask._connectedDevice
	local capture = device and device.captures[ captureId ]
	
	if not capture then
		logger:errorf( "LeicaTetherTask.getCaptureInfo: capture table for captureId %s could not be found.", captureId )
		return {
			status = "error",
			errorMsg = "Could not get capture info for capture id: " .. tostring(captureId),
		}
	end
	
	if capture.captureInfo then
		-- We've already gotten the data
		return {
			status = "ok",
			fileType = LrPathUtils.extension( capture.captureInfo.ifil ),
			fileLength = capture.captureInfo.isiz,
		}
	end
	
--	if not capture.requestedInfo then
--		LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "getCaptureInfo", { capture_id = captureId }, 
--					function( messageType, result ) getCaptureInfoCallback( capture, messageType, result ) end )
--		capture.requestedInfo = true
--	end
	
	return {
		status = "notReady",
	}

end
	

-------------------------------------------------------------------------------

function LeicaTetherTask.startDownload( propertyTable, deviceId, captureId )
	logger:trace( 'LeicaTetherTask.startDownload' )
	
	local device = LeicaTetherTask._connectedDevice
	local capture = device and device.captures[ captureId ]
	if capture then
		LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "startDownload", { capture_id = captureId }, 
					function( messageType, result ) if messageType == "error" then capture.errorCode = result end end )
	end
end

-------------------------------------------------------------------------------

function LeicaTetherTask.getCameraSettings( propertyTable, deviceId, callbackFn )
	logger:trace( 'LeicaTetherTask.getCameraSettings' )
	
	LrRemoteCommunication.sendMessageToServer( LeicaTetherTask._serverConnection, "getCameraSettings", { device_id = deviceId }, 
			function( messageType, result ) 
				logger:trace( 'LeicaTetherTask.getCameraSettings: received result - ', messageType, result )
				
				if messageType == "error" then 
					callbackFn( nil, result.error_code )
				else
					local settings = LeicaTetherUtils.postProcessCameraSettings( result )
					callbackFn( settings )
				end
			end )
end

-------------------------------------------------------------------------------






